'use strict';

/* --------------------------------------------------------------
 payment_methods.js 2018-04-03
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2018 Gambio GmbH
 Release under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Gambio Hub Payment Methods Configuration Controller
 *
 * This module transfers the shop data to the hub payment controller and handles the incoming messages sent via
 * window.postMessage() from the Hub.
 *
 * Add the "data-target" attribute to the #payment-methods-iframe-container element to set the iframe source URL.
 *
 * Add the "data-autostart" attribute to the #payment-methods-iframe-container element to start with  the initialization
 * once the DOM is loaded. Use the "hub:payment_methods:start" event to start the initialization manually.
 *
 * Add the "data-legacy-mode" attribute to the #payment-methods-iframe-container to enable compatibility with older
 * shop pages.
 */
(function () {

	'use strict';

	/**
  * Elements
  *
  * @type {Object}
  */

	var elements = {
		container: document.querySelector('#payment-methods-iframe-container'),
		iframe: null, // Dynamically generated 
		accountContainer: document.querySelector('#account-iframe-container'),
		accountIframe: null, // Dynamically generated
		accountModal: document.querySelector('#account-modal')
	};

	/**
  * "postMessage" message channel
  *
  * @type {Object}
  */
	var messageChannels = {
		authorization: 'Authorization',
		orderStatuses: 'OrderStatuses',
		styleConfig: 'StyleConfig',
		reload: 'Reload',
		configuration: 'Configuration',
		action: 'Action'
	};

	/**
  * URIs
  *
  * @type {Object}
  */
	var uris = {
		shopData: null,
		createSessionKey: null,
		translations: null,
		orderStatuses: null,
		styleConfig: null
	};

	/**
  * Fetched Contents
  *
  * @type {Object}
  */
	var contents = {
		translations: null,
		shopData: null,
		sessionKey: null,
		languageCode: null,
		orderStatuses: null,
		styleConfig: null
	};

	/**
  * Is this an old shop version?
  *
  * @type {Boolean}
  */
	var isLegacyShop = void 0;

	/**
  * Handle AJAX Error
  *
  * This method will throw an exception if the response has the 500 status code.
  *
  * {@link https://developer.mozilla.org/en-US/docs/Web/API/Response/ok}
  *
  * @param {Response} response
  *
  * @return {Response}
  */
	function handleAjaxErrors(response) {
		if (!response.ok) {
			throw new Error(response);
		}

		return response;
	}

	/**
  * Shows a message modal using jQuery UI or native alerts.
  *
  * @param {String} title The modal dialog title.
  * @param {String} content The modal dialog content.
  *
  * {@link http://api.jqueryui.com/1.10/dialog/}
  */
	function showMessage(title, content) {
		var hasJqueryUi = window.$ && window.$.ui;

		if (hasJqueryUi) {
			var $modal = $('<div/>', { class: 'gambio-hub-modal', html: content });

			$modal.dialog({
				title: title,
				buttons: [{ text: 'OK', click: function click() {
						return $modal.dialog('close');
					} }],
				close: function close() {
					return $modal.remove();
				}
			});

			$modal.parents('.ui-dialog').addClass('gx-container');

			var $closeIcon = $modal.parents('.ui-dialog').find('.ui-icon.ui-icon-closethick');

			if ($closeIcon.length) {
				$closeIcon.remove();
			}
		} else {
			alert(title + ' - ' + content);
		}
	}

	/**
  * Fetches the translations and saves the translations response for later reference.
  *
  * @return {Promise}
  */
	function fetchTranslations() {
		var options = { credentials: 'include' };
		var request = new Request(uris.translations, options);

		return window.fetch(request).then(handleAjaxErrors).then(function (response) {
			return response.json();
		}).then(function (parsed) {
			return contents.translations = parsed;
		}).catch(function () {
			return showMessage('Error', 'The page translations couldn\'t be loaded.');
		});
	}

	/**
  * Fetches the shop data and saves the shop data response for later reference.
  *
  * @return {Promise}
  */
	function fetchShopData() {
		var options = { credentials: 'include' };
		var request = new Request(uris.shopData, options);

		return window.fetch(request).then(handleAjaxErrors).then(function (response) {
			return response.json();
		}).then(function (parsed) {
			return contents.shopData = parsed;
		}).catch(function () {
			return showMessage(contents.translations.ERROR, contents.translations.GET_SHOP_DATA_ERROR);
		});
	}

	/**
  * Fetches the existing order statuses.
  *
  * @return {Promise}
  */
	function fetchOrderStatuses() {
		var options = { credentials: 'include' };
		var request = new Request(uris.orderStatuses, options);

		return window.fetch(request).then(handleAjaxErrors).then(function (response) {
			return response.json();
		}).then(function (parsed) {
			return contents.orderStatuses = parsed;
		}).catch(function () {
			return showMessage(contents.translations.ERROR, contents.translations.GET_ORDER_STATUS_ERROR);
		});
	}

	/**
  * Fetches the style edit configurations.
  *
  * @return {Promise}
  */
	function fetchStyleConfig() {
		var options = { credentials: 'include' };
		var request = new Request(uris.styleConfig, options);

		return window.fetch(request).then(handleAjaxErrors).then(function (response) {
			return response.json();
		}).then(function (parsed) {
			return contents.styleConfig = parsed;
		}).catch(function () {
			return showMessage(contents.translations.ERROR, contents.translations.GET_SHOP_DATA_ERROR);
		});
	}

	/**
  * Sends the hub client key to the iframe.
  */
	function sendShopDataToIframe() {
		if (!contents.shopData.clientkey || !contents.sessionKey) {
			elements.iframe.removeEventListener('load', onIframeLoad);
			elements.iframe.src += '&demo';
			return; // No need to continue ...
		}

		var messages = [{
			channel: messageChannels.authorization,
			message: {
				clientKey: contents.shopData.clientkey,
				sessionKey: contents.sessionKey
			}
		}, {
			channel: messageChannels.orderStatuses,
			message: contents.orderStatuses
		}, {
			channel: messageChannels.styleConfig,
			message: contents.styleConfig
		}];

		var queryParameters = location.search.slice(1).split('&');

		var action = queryParameters.filter(function (queryParameter) {
			return queryParameter.indexOf('action=') > -1;
		});

		if (action.length) {
			var split = action[0].split('=').pop().split(':'); // e.g. action=name:value

			messages.push({
				channel: messageChannels.action,
				message: {
					name: split[0],
					value: split[1]
				}
			});

			history.pushState({}, '', jse.core.config.get('appUrl') + '/admin/admin.php?do=HubConfiguration/PaymentMethods');
		}

		elements.iframe.contentWindow.postMessage(messages, '*');
	}

	/**
  * Returns whether the event data contains a message sent from the PostMessageBridge.
  *
  * @param {Event} event Triggered event.
  *
  * @return {Boolean} Is the message a PostMessageBridge message?
  */
	function isPostMessageBridgeMessage(event) {
		return Array.isArray(event.data) && event.data.length && Object.keys(event.data[0]).includes('channel') && Object.keys(event.data[0]).includes('message');
	}

	/**
  * Handles incoming messages.
  *
  * @param {MessageEvent} event Triggered event.
  */
	function onMessage(event) {
		if (!isPostMessageBridgeMessage(event)) {
			return;
		}

		var data = event.data[0];

		if (data.channel.startsWith('Modal') && data.message.startsWith('registration')) {
			if (isLegacyShop) {
				$(elements.accountModal).dialog({
					minWidth: 800,
					minHeight: 700,
					class: 'gx-container'
				});
			} else {
				$(elements.accountModal).modal();
			}
		}

		if (data.channel.startsWith(messageChannels.reload)) {
			window.location.reload();
		}
	}

	/**
  * Performs a request that creates a session key.
  *
  * The creation of a new session requires the existence of a registered client key.
  */
	function createSessionKey() {
		if (!contents.shopData || !contents.shopData.clientkey) {
			return; // Do not start a new session as there's no client key. 
		}

		var options = {
			method: 'POST',
			credentials: 'include'
		};

		var request = new Request(uris.createSessionKey, options);

		return window.fetch(request).then(handleAjaxErrors).then(function (response) {
			return response.json();
		}).then(function (data) {
			return contents.sessionKey = data.gambio_hub_session_key;
		}).catch(function () {
			return showMessage(contents.translations.ERROR, contents.translations.CREATE_SESSION_ERROR);
		});
	}

	/**
  * Handles the iframe's load event.
  */
	function onIframeLoad() {
		window.iFrameResize({
			heightCalculationMethod: 'grow'
		}, '#payment-methods-iframe');

		fetchTranslations().then(fetchShopData).then(fetchOrderStatuses).then(fetchStyleConfig).then(createSessionKey).then(sendShopDataToIframe);
	}

	/**
  * Handles the module initialization.
  */
	function onInit() {
		isLegacyShop = elements.container.getAttribute('data-legacy-mode') !== null;

		contents.languageCode = isLegacyShop ? js_options.global.language_code : window.jse.core.config.get('languageCode');

		uris.translations = isLegacyShop ? 'request_port.php?module=HubConfiguration&action=get_translations' : 'admin.php?do=HubConfigurationAjax/GetTranslations';

		uris.shopData = isLegacyShop ? 'request_port.php?module=HubConfiguration&action=get_shop_data' : 'admin.php?do=HubConfigurationAjax/GetShopData';

		uris.createSessionKey = isLegacyShop ? 'request_port.php?module=HubConfiguration&action=create_session_key' : 'admin.php?do=HubConfigurationAjax/CreateSessionKey';

		uris.orderStatuses = isLegacyShop ? 'request_port.php?module=HubConfiguration&action=get_order_statuses' : 'admin.php?do=HubConfigurationAjax/GetOrderStatuses';

		uris.styleConfig = isLegacyShop ? 'request_port.php?module=HubConfiguration&action=get_style_edit_configuration' : 'admin.php?do=HubConfigurationAjax/GetStyleEditConfiguration';

		elements.iframe = document.createElement('iframe');
		elements.iframe.setAttribute('src', elements.container.getAttribute('data-target'));
		elements.iframe.setAttribute('id', 'payment-methods-iframe');
		elements.iframe.classList.add('hub-iframe');
		elements.iframe.addEventListener('load', onIframeLoad);
		elements.container.appendChild(elements.iframe);
		elements.container.removeAttribute('data-target');
		elements.container.removeAttribute('data-autostart');
		elements.container.removeAttribute('data-legacy-mode');

		var event = document.createEvent('Event');
		event.initEvent('hub:account:start', true, false);
		window.dispatchEvent(event);
	}

	var initEvent = elements.container.getAttribute('data-autostart') !== null ? 'DOMContentLoaded' : 'hub:payment_methods:start';
	window.addEventListener(initEvent, onInit);
	window.addEventListener('message', onMessage);
})();
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
